/***************************************************************************
*
* Copyright 2010,2011 BMW Car IT GmbH
*
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*        http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
****************************************************************************/
#include "SurfaceRemoveCommand.h"
#include "ICommandExecutor.h"
#include "Scene.h"
#include "Log.h"
#include <sstream>
#include "Surface.h"

ExecutionResult SurfaceRemoveCommand::execute(ICommandExecutor* executor)
{
    Scene& scene = *(executor->getScene());

    ExecutionResult result = ExecutionFailed;

    Surface* surface = scene.getSurface(m_idToRemove);
    if (surface)
    {
        ilmObjectNotification objNotification = {ILM_SURFACE,m_idToRemove,ILM_FALSE};
        RendererListConstIterator it;

        executor->addNewObjectNotification(objNotification);
        RendererList *rList = executor->getRendererList();
        SeatList surfAccpSeats = surface->getAcceptedSeats();
        SeatList::iterator seat_it;

        // Tell the InputManager to remove all focuses that are currently
        // associated with this surface
        result = ExecutionSuccess;
        for (seat_it = surfAccpSeats.begin();
                (seat_it != surfAccpSeats.end()) && (result == ExecutionSuccess); seat_it++)
        {

            for (it = rList->begin(); it != rList->end(); ++it)
            {
                InputManager* pInputManager = (*it)->getInputManager((char*)(*seat_it)->c_str());
                if (NULL != pInputManager)
                {
                    pInputManager->dropFocuses(m_idToRemove);
                }
                else
                {
                    std::string seatName((*seat_it)->c_str());
                    result = ExecutionFailed;
                    LOG_ERROR("SurfaceRemoveCommand", "obtained NULL input manager for seat :" << seatName);
                }
            }
        }

        // Removing the client buffers which were created from BaseWindowSystem in the case of shm buffer usage
        // if we remove only the ilmsurface here, then the actual client buffer allocated in BaseWindowSystem
        // will not be freed, so it will lead to memory leak
        for (it = rList->begin(); it != rList->end(); ++it)
        {
            (*it)->destroyClientBufferShm(surface);
        }


        result = scene.removeSurface(surface) ? ExecutionSuccessRedraw : ExecutionSuccess;
    }

    return result;
}

const std::string SurfaceRemoveCommand::getString()
{
    std::stringstream description;
    description << "SurfaceRemoveCommand("
                << "idToRemove=" << m_idToRemove << "(0x" << std::hex << m_idToRemove << ")" << std::dec
                << ")";
    return description.str();
}
